#ifndef __CWindowAttributes__
#define __CWindowAttributes__

#include <Basics/CString.hpp>
#include "../Basics/CColour.hpp"
#include "../Graphics/CWindowIcon.hpp"

//	===========================================================================

using Exponent::Basics::CString;
using Exponent::GUI::Basics::CColour;
using Exponent::GUI::Graphics::CWindowIcon;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Windowing
		{
			/**
			 * @class CWindowAttributes CWindowAttributes.hpp
			 * @brief Attributes of a window
			 *
			 * @date 18/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CWindowAttributes.hpp,v 1.6 2007/02/08 21:07:54 paul Exp $
			 */
			class CWindowAttributes : public CCountedObject
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================

			public:

//	===========================================================================

				/**
				 * @enum EWindowStyle
				 * @brief Enumerated default styles of window
				 */
				enum EWindowStyle
				{
					e_default = 0,							/**< Default, we can size, max, min, close etc */
					e_noMaximiseNoSize,						/**< Can't size, cant maximise, can close, can minise */
					e_noMaximiseNoSizeNoClose,				/**< Cant size, cant maximise, cant close, can minimise */
				};

//	===========================================================================

				/**
				 * Construction
				 */
				CWindowAttributes();

				/**
				 * Destruction
				 */
				virtual ~CWindowAttributes();

//	===========================================================================

				/**
				 * Assignment operator
				 * @param attributes The attributes to copy
				 * @retval CWindowAttributes& A refernce to this
				 */
				CWindowAttributes &operator = (const CWindowAttributes &attributes);

//	===========================================================================

				/**
				 * Intialise with a style
				 * @param style The style to initialise with
				 */
				void initialise(const EWindowStyle style);

				#ifdef WIN32
					/**
					 * Setup the core window values
					 * @param style The style
					 * @param styleEx The ex style
					 * @see CreateWindowEx in patform SDK
					 */
					void initialise(const long style, const long styleEx);
				#else
					/**
					 * Setup the core window values
					 * @param style The style of window
					 * @param windowClass The class of window
					 * @see CreateWindow in the platform SDK
					 */
					void initialise(WindowAttributes style, WindowClass windowClass);	
				#endif

//	===========================================================================

				/**
				 * Set the attributes of the window
				 * @param title The title of the window
				 * @param backgroundColour The background colour of the window
				 * @param alpha The alpha value of the window 0 - 1
				 * @param windowIcon The icon for the window
				 * @param enabled The enable status of the window, true if enabled, false if disabled
				 * @param shadow Does the window have a shadow?
				 */
				void setWindowAttributes(const CString &title, const CColour &backgroundColour, const double alpha, CWindowIcon *windowIcon, const bool enabled, const bool shadow);

//	===========================================================================

				/**
				 * Set the window title
				 * @param title The title of the window
				 */
				void setWindowTitle(const CString &title) { m_title = title; }

				/**
				 * Set the background colour
				 * @param backgroundColour The background colour of the window
				 */
				void setBackgroundColour(const CColour &backgroundColour) { m_backgroundColour = backgroundColour; }

				/**
				 * Set the alpha value of the window
				 * @param alpha The alpha value of the window 0 - 1
				 */
				void setWindowAlphaValue(const double alpha) { m_alpha = alpha; }

				/**
				 * Set the window icon
				 * @param icon The icon for the window
				 */
				void setWindowIcon(CWindowIcon *icon);

				/**
				 * Set the window enabled
				 * @param enabled The enable status of the window, true if enabled, false if disabled
				 */
				void setWindowEnabled(const bool enabled) { m_enabled = enabled; }

				/**
				 * Set the window shadow useage
				 * @param shadow Does the window have a shadow?
				 */
				void windowHasShadow(const bool shadow) { m_hasShadow = shadow; }

//	===========================================================================

				/**
				 * Get the style
				 * @retval long The platform specific window style
				 */
				long getWindowStyle() const { return m_windowStyle; }

				/** @cond */
				#ifdef WIN32	
				/** @endcond */
					/**
					 * Get the window class
					 * @retval long The windows class EX.
					 */
					long getWindowStyleEx() const { return m_windowStyleEx; }	
				/** @cond */		
				#else
				/** @endcond */
					/**
					 * Get the window class
					 * @retval WindowClass The class of window
					 */
					WindowClass getWindowClass() const { return m_windowClass; }
				/** @cond */
				#endif
				/** @endcond */

				/**
				 * Get the window title
				 * @retval const CString& The title of the window
				 */
				const CString &getWindowTitle() const { return m_title; }

				/**
				 * Get the background colour
				 * @retval const CColour& The background colour
				 */
				const CColour &getBackgroundColour() const { return m_backgroundColour; }

				/**
				 * Get the window alpha value
				 * @retval double The alpha level 0 - 1
				 */
				double getWindowAlphaValue() const { return m_alpha; }

				/**
				 * Get the window alpha value
				 * @retval long Window alpha in range 0 - 255
				 */
				long getWindowAlphaValueLong() const { return (long)(m_alpha * 255.0); }

				/**
				 * Get the window icon
				 * @retval CWindowIcon* The icon for the window, may be NULL
				 */
				CWindowIcon *getWindowIcon() const { return m_windowIcon; }

				/**
				 * Is the window enabled?
				 * @retval bool True if window is enabled, false otherwise
				 */
				bool isWindowEnabled() const { return m_enabled; }

				/**
				 * Does the window have a shadow?
				 * @retval bool True if window has shadow, false otherwise
				 */
				bool windowHasShadow() const { return m_hasShadow; }

//	===========================================================================

			protected:

//	===========================================================================

				CString m_title;							/**< Title of the window */
				CColour m_backgroundColour;					/**< Background colour of the window */
				double m_alpha;								/**< Alpha value for the window */
				CWindowIcon *m_windowIcon;					/**< The icon of the window */
				bool m_enabled;								/**< Is the window enabled */
				bool m_hasShadow;							/**< Does the window have a shadow? */

//	===========================================================================

				/** @cond */
				#ifdef WIN32
				/** @endcond */
					long m_windowStyle;						/**< Style of the window */
					long m_windowStyleEx;					/**< Extended style of the window */
				/** @cond */
				#else
				/** @endcond */
					WindowAttributes m_windowStyle;			/**< The attributes / style of the window */
					WindowClass m_windowClass;				/**< The class of the window */
				/** @cond */
				#endif
				/** @endcond */

//	===========================================================================

			};
		}
	}
}
#endif	// End of CWindowAttributes.hpp